# http://openmidiproject.sourceforge.jp/documentations.html

#-------------------------------------------------------------------------------
#
#	単旋律クラス
#
class Track < Array

	@@next_ch = -1
	@@KEYS = Hash.new; n = -1
	['C', 'C#', 'D', 'D#', 'E', 'F', 'F#', 'G', 'G#', 'A', 'A#', 'B'].each {|k|
		@@KEYS[k] = (n += 1)
	}

	def initialize(timebase = 240, ch = nil)
		@timebase = timebase									# タイムベース
		@delta = 0
		@last = nil
		if(ch)													# チャンネル指定
			@ch = (ch - 1) % 16
		else
			@ch = (@@next_ch += 1) % 15							# チャンネル自動採番
			@ch += 1 if(@ch > 8)								# リズムチャンネルを回避
		end
	end

	def push_delta												# デルタタイムをエンコード
		if(@delta != 0)
			tmp = Array.new
			loop {
				tmp.unshift(0x80 + @delta % 0x80)
				break if((@delta >>= 7) == 0)
			}
			tmp[tmp.size - 1] -= 0x80
			self.push(*tmp)
			@delta = 0
		else
			self.push(0)
		end
	end

	def time(d)
		@delta += d
	end

	def dtime(d)
		@delta += 4 * @timebase / d
	end

	def panpot(p)
		push_delta()
		self.push(0xb0 + @ch, 0x0a, p)
	end

	def tone(t)													# 音色変更
		push_delta()
		self.push(0xc0 + @ch, t)
	end

	def note(k, v = 96)
		note0(48 + 12 * k[0, 1].to_i + @@KEYS[k[1, 9]], v)
#		note0(12 + 12 * k[0, 1].to_i + @@KEYS[k[1, 9]], v)
	end

	def note0(n, v = 96)
		off
		push_delta()
		self.push(0x90 + @ch, n, v)
		@last = n
	end

	def off
		if(@last)
			push_delta()
			self.push(0x80 + @ch, @last, 0)
		end
	end

	def close
		push_delta()
		self.push(0xff, 0x2f, 0x00)								# トラック完了コード
		size = self.size
		(0..3).each {
			self.unshift(size % 0x100)							# トラックのバイト数
			size >>= 8
		}
		self.unshift(*'MTrk'.unpack('c*'))						# 固定値
	end

	def each
		yield(self)
	end

# TODO
#  音長の設定により自動的にノートオフする機能

end

#-------------------------------------------------------------------------------
#
#	伴奏クラス
#
class Chord_track < Array

	def initialize(timebase = 240, ch = nil)
		@timebase = timebase									# タイムベース
		(0..5).each {|t|
			self.push(Track.new(@timebase, ch))
			self[t].panpot([1, 21, 41, 81, 101, 128][t]) if(ch != 10)
			self[t].tone(25) if(ch != 10)						# 25: Acoustic Guitar(steel)
		}
	end

	def chord_pattern(opt = Hash[])
		@pattern = opt['pattern'] ? opt['pattern'] : 'stroke-1'
		@length  = opt['length']  ? opt['length']  : 4			# n拍子
		@volume  = opt['volume']  ? opt['volume']  : 100
		@sdelay  = opt['sdelay']  ? opt['sdelay']  : 3			# ストロークの速さ
	end

	def chord(chord)
		#
		#	ストローク
		#
		if(@pattern =~ /stroke-(\d+)/)
			@@STROKE_VLC = [
				[nil, nil, nil, nil],
				[ 80,  60,  70,  60],							# 4beat
				[ 80, -60,  70, -60,  80, -60,  70, -60],		# 8beat
				[ 80, nil,  70, -60, nil, -60,  70, -60],
				[ 80, -60,  60, -60,  70, -60,  60, -60,  70, -60,  60, -60,  70, -60,  60, -60],	# 16beat
				[ 80, nil,  60, nil,  60, nil,  70, -60,  80, -60,  60, nil,  60, nil,  70, -60],
			]
			vlc = @@STROKE_VLC[$1.to_i]
			(0...(vlc.size * @length / 4)).each {|n|
				v = vlc[n % vlc.size]
				(0..5).each {|t|
					self[t].time((v > 0 ? t : 5 - t) *  @sdelay) if(v)
					self[t].note(chord[t], v.abs * @volume / 100) if(chord[t] and v)
					self[t].dtime(vlc.size)
					self[t].time((v > 0 ? t : 5 - t) * -@sdelay) if(v)
				}
			}
		#
		#	アルペジオ
		#
		elsif(@pattern =~ /arpeggio-(\d+)/)
			@@ARPEGGIO_SEQ = [
				[nil, nil, nil, nil],
				[[9], [3, 2, 1], [3, 2, 1], [3, 2, 1]],			# 9: Base String
				[[9], [3], [1, 2], [3]],
				[[9], [2], [1], [2], [3], [2], [1], [2]],
				[[9, 1], nil, [3], [2], [9], [1], [3], nil],	# 3-finger
				[[9, 3], [2], [1], nil],						# for triple time
				[[9, 3], [1], [2], nil],
				[[9, 3], [2], [1], [9, 3], [2], [1], nil, nil],
				[[9, 3], [2], [1], nil],						# for triple time
				[[9, 3], [], [], [], [3], [], [], [2], [1], [], [], [], nil, nil, nil, nil],
			]
			seq = @@ARPEGGIO_SEQ[$1.to_i]
			rchord = chord.reverse
			(0...(seq.size * @length / 4)).each {|n|
				seq[n % seq.size].each {|ss| s = ss - 1
					if(rchord[s])
						self[s].note(rchord[s], 50 * @volume / 100)
					else
						self[rchord.size - 1].note(rchord.last, 50 * @volume / 100)
					end
				} if(seq[n % seq.size])
				(0..5).each {|t|
					self[t].dtime(seq.size)
				}
			}
		end
	end

	def off
		self.each {|t| t.off }
	end

	def close
		self.each {|t| t.close }
	end
end

#-------------------------------------------------------------------------------
#
#	リズムクラス
#
class Rhythm_track < Chord_track

	def rhythm_pattern(opt = Hash[])
		opt['pattern'] ||= 'rock-1'
		if(opt['volume'].class.to_s != 'Hash')
			vol = Hash[38, 60, 44, 50, 0, 90]					# default: 38:Snare->60, 44:Hi-Hat->50, 0:Other->90
			vol.each {|k, v|
				vol[k] = v * opt['volume'] / 100
			} if(opt['volume'].class.to_s == 'Fixnum')
			opt['volume'] = vol
		end
		chord_pattern(opt)
	end

	def rhythm(opt = Hash[])
		#
		# リズムパターン
		#
		@@RHYTHM = Hash[
			# 35: Acoustic Bass Drum
			# 37: Side Stick
			# 38: Acoustic Snare
			# 47: Low-Mid Tom
			# 44: Pedal Hi-Hat
			# 81: Open Triangle

			'rhythm-0',	[nil, nil, nil, nil],

			'waltz-1',	[[35], [38], [38], nil],

			'rumba-1',	[[44,35,47], nil, [44],    [44], [44,38], nil, [44,47], nil,
						 [44,35],    nil, [44,47], nil,  [44,35], nil, [44],    nil,
						 [44,35],    nil, [44],    [44], [44,38], nil, [44],    nil,
						 [44,35,47], nil, [44,38], nil,  [44,38], nil, [44,38], nil],

			'swing-1',	[[44], nil, nil, nil, [44], nil, nil, [44]],

			'rock-1',	[[44,47], [44],    [44,38], [44], [44,47], [44], [44,38], [44],
						 [44],    [44,47], [44,38], [44], [44,47], [44], [44,38], [44,47]],

			'pops-1',	[[44,35], [44], [44,38], [44,35]],

			'disco-1',	[[44,38], [44,35], [44,35], [44], [44,38], [44,35], [44,35], [44]],

			'samba-1',	[[44,38], [44], [44,47], [44,35], [44,35], [44], [44,47], [44,35,47]],

			'metro-1',	[[37,81], [37], [37], [37]],
		]
		seq = @@RHYTHM[@pattern]
		(0...(seq.size * @length / 4)).each {|n|
			t = -1; seq[n % seq.size].each {|s|
				self[t += 1].note0(s, @volume[s] ? @volume[s] : @volume[0])
			} if(seq[n % seq.size])
			(0..5).each {|t|
				self[t].dtime(seq.size)
			}
		}
	end
end

#-------------------------------------------------------------------------------
#
#	Midiファイルクラス
#
class Midi < Array

	def initialize(timebase)
		@timebase = timebase									# タイムベース
		@nTrack = 0												# トラック数
	end

	def tempo(bpm)
		t = Track.new(@timebase)
		us = 60000000 / bpm
		(0..2).each {
			t.unshift(us % 0x100)
			us >>= 8
		}
		t.unshift(0x00, 0xff, 0x51, 0x03)
		t.close
		self.track(t)
	end

	def track(t)
		t.each {|tt|
			@nTrack += 1
			self.push(*tt)
		}
	end

	def close
		self.unshift(@timebase >> 8, @timebase % 0x100)			# タイムベース
		self.unshift(0x00, @nTrack)								# トラック数
		self.unshift(0x00, 0x01)								# フォーマット(複数トラック)
		self.unshift(0x00, 0x00, 0x00, 0x06)					# ヘッダのバイト数
		self.unshift(*'MThd'.unpack('c*'))						# 固定値
	end

	def write(fn)
		open(@filename = fn, 'w') {|fh|
			fh.write(self.pack('c*'))
		}
	end

	def play(fn = @filename)
		system("timidity #{fn}") if(fn)
	end
end

